﻿using System;
using System.Web.Mvc;
using System.Linq;
using MVC_App.Models;
using MVC_App.Validation;

namespace MVC_App.Controllers
{
	public class HomeController : Controller
	{
		[HttpGet]
		public ActionResult EditOrder()
		{
			return View(BuildDummyBestellung());
		}

		static BestellungForm BuildDummyBestellung()
		{
			return new BestellungForm
					{
						Lieferadresse = new LieferadresseForm(),
						Kreditkarte = new KreditkarteForm(),
						Pizza = new PizzaForm
											{
												Größe = PizzaGrößen.Mittel,
											}
					};
		}

		[HttpPost]
		public ActionResult SubmitOrder(BestellungForm bestellung)
		{
			DateTime? lieferzeitpunkt = ValidateLieferzeitpunkt(bestellung);

			ZahlungModel zahlung = ValidateZahlung(bestellung);

			GetränkModel getränk = null;
			if (bestellung.HasGetränk)
				getränk = ValidateGetränk(bestellung);

			PizzaModel pizza = ValidatePizza(bestellung);

			DessertModel dessert = null;
			if (bestellung.HasDessert)
				dessert = ValidateDessert(bestellung);

			if (ModelState.IsValid
				&& lieferzeitpunkt != null
				&& zahlung != null
				&& !(bestellung.HasGetränk && getränk == null)
				&& pizza != null
				&& !(bestellung.HasDessert && dessert == null))
			{
				var orderModel = new OrderModel(lieferzeitpunkt.Value, zahlung, getränk, pizza, dessert);
				return View("Success", orderModel);
			}

			return View("EditOrder", bestellung);
		}

		private DateTime? ValidateLieferzeitpunkt(BestellungForm bestellung)
		{
			Time time;
			switch (bestellung.LieferzeitpunktArt)
			{
				case Lieferzeitpunkte.Sofort:
					return DateTime.Now;

				case Lieferzeitpunkte.Heute:
					time = ValidateUhrzeit(bestellung);
					if (time == null)
						return null;

					return DateTime.Now.SetTime(time);

				case Lieferzeitpunkte.Am:
					time = ValidateUhrzeit(bestellung);
					var date = ValidateDatum(bestellung);

					if (time == null || date == null)
						return null;

					return date.Value.SetTime(time);

				default:
					return null;
			}
		}

		private Time ValidateUhrzeit(BestellungForm bestellung)
		{
			if (bestellung.Lieferzeitpunkt.Uhrzeit != null)
			{
				var result = Time.Build(bestellung.Lieferzeitpunkt.Uhrzeit);
				if (result != null)
					return result;
			}

			ModelState.AddModelError("Lieferzeitpunkt.Uhrzeit", "Bitte geben Sie eine gültige Uhrzeit für die Lieferung an.");
			return null;
		}

		private DateTime? ValidateDatum(BestellungForm bestellung)
		{
			if (bestellung.Lieferzeitpunkt.Datum != null)
			{
				var result = DateTimeEx.Build(bestellung.Lieferzeitpunkt.Datum);
				if (result != null)
					return result;
			}

			ModelState.AddModelError("Lieferzeitpunkt.Datum", "Bitte geben Sie ein gültiges Datum für die Lieferung an.");
			return null;
		}

		private ZahlungModel ValidateZahlung(BestellungForm bestellung)
		{
			switch (bestellung.Zahlmodus)
			{
				case Zahlmodi.Haustür:
					return new HaustürZahlungModel();
				case Zahlmodi.Kreditkarte:
					return ValidateKreditkarteZahlung(bestellung);
				default:
					ModelState.AddModelError("Zahlmodus", "Bitte wählen Sie eine Zahlungmethode aus");
					return null;
			}
		}

		private KreditkarteZahlungModel ValidateKreditkarteZahlung(BestellungForm bestellung)
		{
			// Erste Validierungsebene: Vorhandensein von Daten
			bool nullError = false;

			if (string.IsNullOrWhiteSpace(bestellung.Kreditkarte.Name))
			{
				nullError = true;
				ModelState.AddModelError("Kreditkarte.Name", "Bitte geben Sie den Namen des Kreditkarteninhabers ein");
			}

			if (string.IsNullOrWhiteSpace(bestellung.Kreditkarte.Nummer))
			{
				nullError = true;
				ModelState.AddModelError("Kreditkarte.Nummer", "Bitte geben Sie die Nummer der Kreditkarten ein");
			}

			if (string.IsNullOrWhiteSpace(bestellung.Kreditkarte.GültigBisMonat))
			{
				nullError = true;
				ModelState.AddModelError("Kreditkarte.GültigBisMonat", "Bitte geben Sie den Ablaufmonat der Kreditkarte an");
			}

			if (string.IsNullOrWhiteSpace(bestellung.Kreditkarte.GültigBisJahr))
			{
				nullError = true;
				ModelState.AddModelError("Kreditkarte.GültigBisJahr", "Bitte geben Sie das Ablaufjahr der Kreditkarte an");
			}

			if (!nullError)
			{
				// Zweite Validierungsebene: Korrektheit der zusammensetzbaren Daten.
				// Hier: Monat und Jahr.
				var dateError = false;
				int monat;
				int jahr;

				if (!int.TryParse(bestellung.Kreditkarte.GültigBisMonat, out monat)
					|| monat < 1
					|| monat > 12)
				{
					dateError = true;
					ModelState.AddModelError("Kreditkarte.GültigBisMonat", "Bitte geben Sie einen gültigen Ablaufmonat für die Kreditkarte an");
				}

				if (!int.TryParse(bestellung.Kreditkarte.GültigBisJahr, out jahr)
					|| jahr < 2012
					|| jahr > 2020)
				{
					dateError = true;
					ModelState.AddModelError("Kreditkarte.GültigBisJahr", "Bitte geben Sie einen gültiges Ablaufjahr für die Kreditkarte an");
				}

				if (!dateError)
				{
					var gültigkeit = new DateTime(jahr, monat, 1);

					// Dritte Validierungsebene: Semantische Korrektheit des gesamten Typs
					var result = KreditkarteServiceGateway.CheckValidity(
						bestellung.Kreditkarte.Name,
						bestellung.Kreditkarte.Nummer,
						gültigkeit);

					if (result == null)
					{
						ModelState.AddModelError("Kreditkarte.Name", "Die Kreditkarteninformationen sind ungültig");
						ModelState.AddModelError("Kreditkarte.Nummer", "");
						ModelState.AddModelError("Kreditkarte.GültigBisMonat", "");
						ModelState.AddModelError("Kreditkarte.GültigBisJahr", "");
					}
				}
			}

			return null;
		}

		private static DateTime BuildDateTime(int monat, int jahr)
		{
			try
			{
				return new DateTime(jahr, monat, 1);
			}
			catch (Exception)
			{
				throw new ValidationException("Ungültige Datumsangabe");
			}
		}

		private GetränkModel ValidateGetränk(BestellungForm bestellung)
		{
			if (bestellung.Getränk.Name == null)
				ModelState.AddModelError("Getränk.Name", "Bitte wählen Sie ein gültiges Getränk aus");

			int größe = -1;
			if (bestellung.Getränk.Flaschengröße == null
				|| !int.TryParse(bestellung.Getränk.Flaschengröße, out größe))
				ModelState.AddModelError("Getränk.Flaschengröße", "Bitte wählen Sie eine gültige Größe aus");

			if (bestellung.Getränk.Name != null && größe != -1)
			{
				var getränkSorte = GetränkSorteRepository.Find(bestellung.Getränk.Name);
				if (getränkSorte == null)
					ModelState.AddModelError("Getränk.Name", "Bitte wählen Sie ein gültiges Getränk aus");
				else
				{
					if (!getränkSorte.VerfügbareVolumen.Contains(größe))
						ModelState.AddModelError("Getränk.Flaschengröße",
						                         "Das von Ihnen gewählte Getränk ist nicht in dieser Größe erhältlich");
					else
						return new GetränkModel(getränkSorte, bestellung.Getränk.Flaschengröße, bestellung.Getränk.Gekühlt);
				}
			}
			return null;
		}

		private PizzaModel ValidatePizza(BestellungForm bestellung)
		{
			bool error = false;
			if (!bestellung.Pizza.Größe.HasValue)
			{
				ModelState.AddModelError("Pizza.Größe", "Bitte wählen Sie eine gültige Pizzagröße aus");
				error = true;
			}
			if (string.IsNullOrWhiteSpace(bestellung.Pizza.Sorte))
			{
				ModelState.AddModelError("Pizza.Sorte", "Bitte wählen Sie eine gültige Pizzasorte aus");
				error = true;
			}

			if (error)
				return null;

			if (!PizzaRepository.Sorten.Contains(bestellung.Pizza.Sorte))
			{
				ModelState.AddModelError("Pizza.Sorte", "Bitte wählen Sie eine gültige Pizzasorte aus");
				return null;
			}

			return new PizzaModel(bestellung.Pizza.Sorte, bestellung.Pizza.Größe.Value);
		}

		private DessertModel ValidateDessert(BestellungForm bestellung)
		{
			// Wenn das DessertForm keine Nutzdaten hat wird es garnicht erst erstellt.
			// Dann macht es hier Bumm!
			if (!bestellung.Dessert.Name.HasValue)
			{
				ModelState.AddModelError("Dessert.Name", "Bitte wählen Sie ein gültiges Dessert aus");
				return null;
			}
			return new DessertModel(bestellung.Dessert.Name.Value);
		}
	}
}
